import { block, game_loop, maxX, maxY } from ".";

export const block_shape: { shape: number[][], color: number }[] = [
    { shape: [[1, 1, 1, 1]], color: 1 },
    { shape: [[1, 1], [1, 1]], color: 2 },
    { shape: [[1, 1, 1], [0, 1, 0]], color: 3 },
    { shape: [[1, 1, 1], [1, 0, 0]], color: 4 },
    { shape: [[1, 1, 1], [0, 0, 1]], color: 5 },
    { shape: [[1, 1, 0], [0, 1, 1]], color: 6 },
    { shape: [[0, 1, 1], [1, 1, 0]], color: 3 }
];

export const game_board = new Map<string, number[][]>();
const currentPiece_g = new Map<string, { shape: number[][], color: number }>();
export const game_X = new Map<string, number>();
export const game_Y = new Map<string, number>();
export const score = new Map<string,number>();

export function newPiece(id:string) {
    const piece = block_shape[Math.floor(Math.random() * block_shape.length)];
    currentPiece_g.set(id, piece);
    game_X.set(id, Math.floor(maxX / 2) - Math.floor(piece.shape[0].length / 2));
    game_Y.set(id, 0);
    
    if(!canMove(0, 0, id)) {
        game_over(id); 
    }
}

export function game_over(id:string) {
    clearInterval(game_loop.get(id));
}

export function canMove(offsetX: number, offsetY: number, id:string): boolean {
    const currentPiece = currentPiece_g.get(id);
    const currentX = game_X.get(id) as number;
    const currentY = game_Y.get(id) as number;
    if (!currentPiece) return false;
  
    for (let y = 0; y < currentPiece.shape.length; y++) {
        for (let x = 0; x < currentPiece.shape[y].length; x++) {
            if (currentPiece.shape[y][x]) {
                const board = game_board.get(id) as number[][];
                const newX = currentX + x + offsetX;
                const newY = currentY + y + offsetY;
                if (newX < 0 || newX >= maxX || newY >= maxY || (newY >= 0 && board[newY][newX])) {
                    return false;
                }
            }
        }
    }
    return true;
}

function lockPiece(id:string) {
    const currentPiece = currentPiece_g.get(id);
    const currentX = game_X.get(id) as number;
    const currentY = game_Y.get(id) as number;
    if (!currentPiece) return;

    const board = game_board.get(id) as number[][];
    for (let y = 0; y < currentPiece.shape.length; y++) {
        for (let x = 0; x < currentPiece.shape[y].length; x++) {
            if (currentPiece.shape[y][x]) {
                board[currentY + y][currentX + x] = currentPiece.color;
            }
        }
    }
    game_board.set(id, board);
    clearLines(id);
    newPiece(id);
}

function clearLines(id:string) {
    const board = game_board.get(id) as number[][];
    for (let y = maxY - 1; y >= 0; y--) {
        if (board[y].every(cell => cell !== 0)) {
            board.splice(y, 1);
            board.unshift(Array(maxX).fill(0));
            score.set(id, (score.get(id) as number) + 100);
        }
    }
    game_board.set(id, board);
}

function draw(id:string) {
    const currentPiece = currentPiece_g.get(id);
    const currentX = game_X.get(id) as number;
    const currentY = game_Y.get(id) as number;
    const board = game_board.get(id) as number[][];
    const display: number[][] = board.map(row => [...row]);
    if (currentPiece) {
        for (let y = 0; y < currentPiece.shape.length; y++) {
            for (let x = 0; x < currentPiece.shape[y].length; x++) {
                if (currentPiece.shape[y][x]) {
                    display[currentY + y][currentX + x] = currentPiece.color;
                }
            }
        }
    }
    return display.map(row => row.map(cell => block[cell]).join('')).join('\n')
}

export function gameLoop(id:string) {
    if (canMove(0, 1, id)) {
        game_Y.set(id, game_Y.get(id) as number + 1);
    } else {
        lockPiece(id);
    }
    return draw(id);
}

export function tetris_move(key:string, id:string) {
    const currentPiece = currentPiece_g.get(id);
    if (key === 'left' && canMove(-1, 0, id)) {
        game_X.set(id, game_X.get(id) as number - 1);
    } else if (key === 'right' && canMove(1, 0, id)) {
        game_X.set(id, game_X.get(id) as number + 1);
    } else if(key === 'down' && canMove(0, 1, id)) {
        game_Y.set(id, game_Y.get(id) as number + 1);
    } else if (key === 'rotate' && currentPiece) {
        const rotated = { 
            shape: currentPiece.shape[0].map((_, i) => currentPiece.shape.map(row => row[i]).reverse()),
            color: currentPiece.color
        };
        if (canMove(0, 0, id)) {
            currentPiece_g.set(id, rotated);
        }
    }
    return draw(id);
}