import { Canvas } from "canvas";
const styles = {
    circle: {
        margin: 5,
        fillColor: '#fff',
        strokeColor: '#000',
        strokeWidth: 2
    },
    options: {
        font: '14px Sans',
        textBaseline: 'middle',
        portionStrokeColor: '#000',
        portionStrokeWidth: 1
    },
    centerDot: {
        fillColor: '#000',
        radius: 5
    },
    arrow: {
        height: 20,
        width: 10,
        fillColor: '#fff',
        strokeColor: '#000',
        strokeWidth: 2
    }
} as const;

export function spin_create(candidates:string[], canvas:Canvas, ctx:CanvasRenderingContext2D, result:number, spin_max:number) {
    const cw = canvas.width;
    const ch = canvas.height;
    const cx = cw / 2;
    const cy = ch / 2;
    const radius = Math.min(cw, ch) / 2 - styles.circle.margin;
    const endAngle = (result * Math.PI) / 180;
    const portionAngle = (Math.PI * 2) / candidates.length;
    const optionOffset = portionAngle / 2;
    
    const colors = get_color(candidates.length);
    
    const easing = (step: number) => {
        return -endAngle * ((step = step / spin_max - 1) * step) + endAngle;
    }
    
    const drawCicle = () => {
        ctx.beginPath();
        ctx.arc(0, 0, radius, 0, Math.PI * 2);
        ctx.fillStyle = styles.circle.fillColor;
        ctx.strokeStyle = styles.circle.strokeColor;
        ctx.lineWidth = styles.circle.strokeWidth;
        ctx.fill();
        ctx.stroke();
        
        (ctx.setTransform as any)(1, 0, 0, 1, 0, 0);
    };
    
    const drawOption = (
        angle: number,
        index: number,
        option: string,
        backgroundColor: string
    ) => {
        const optionAngle = angle + portionAngle * index;
        const textColor = get_background(backgroundColor);
        
        const drawPortion = () => {
            ctx.beginPath();
            ctx.fillStyle = backgroundColor;
            ctx.strokeStyle = styles.options.portionStrokeColor;
            ctx.lineWidth = styles.options.portionStrokeWidth;
            ctx.moveTo(cx, cy);
            ctx.arc(
                cx,
                cy,
                radius,
                optionAngle - optionOffset,
                optionAngle + optionOffset,
                false
            );
            ctx.lineTo(cx, cy);
            ctx.fill();
            ctx.stroke();
        };
        
        const drawText = () => {
            ctx.save();
            ctx.translate(cx, cy);
            ctx.rotate(optionAngle);
            ctx.fillStyle = textColor;
            ctx.font = styles.options.font;
            ctx.textBaseline = styles.options.textBaseline;
            
            const textWidth = Math.min(ctx.measureText(option).width, radius);
            const centeredTextX = (radius - textWidth) / 2;
            
            ctx.fillText(option, centeredTextX, 0, radius);
            ctx.restore();
        };
        
        drawPortion();
        drawText();
    };
    
    const drawOptions = (angle: number) => {
        const [first, ...rest] = candidates;
        const orderedOptions = [first, ...rest.reverse()];
        
        const [firstColor, ...restColors] = colors;
        const orderedColors = [firstColor, ...restColors.reverse()];
        
        for (let i = 0; i < orderedOptions.length; i++) {
            const option = orderedOptions[i];
            const color = orderedColors[i];
            drawOption(angle, i, option, color);
        }
    };
    
    const drawCenterDot = () => {
        ctx.beginPath();
        ctx.arc(cw / 2, ch / 2, styles.centerDot.radius, 0, Math.PI * 2);
        ctx.fillStyle = styles.centerDot.fillColor;
        ctx.fill();
    };
    
    const drawArrow = () => {
        const initialX = cw - styles.circle.margin;
        const initialY = cy - styles.arrow.width;
        ctx.beginPath();
        ctx.moveTo(initialX, initialY);
        ctx.lineTo(initialX, initialY + styles.arrow.width * 2);
        ctx.lineTo(initialX - styles.arrow.height, initialY + styles.arrow.width);
        ctx.closePath();
        ctx.fillStyle = styles.arrow.fillColor;
        ctx.strokeStyle = styles.arrow.strokeColor;
        ctx.lineWidth = styles.arrow.strokeWidth;
        ctx.fill();
        ctx.stroke();
    };
    
    const draw = (step: number): void => {
        const angle = easing(step);
        
        ctx.translate(cx, cy);
        ctx.rotate(angle);
        ctx.clearRect(0, 0, cw, ch);
        
        drawCicle();
        drawOptions(angle);
        drawCenterDot();
        drawArrow();
    };
    
    const rotate = (step: number): void => {
        draw(step);
    };
    
    const getOptionByStep = (step: number) => {
        const angle = easing(step) + optionOffset;
        const optionIndex = Math.floor(angle / portionAngle) % candidates.length;
        return {
            option: candidates[optionIndex],
            color: colors[optionIndex]
        };
    };
    
    const getColorByStep = (step: number) => {
        const angle = easing(step) + optionOffset;
        const optionIndex = Math.floor(angle / portionAngle) % candidates.length;
        return {
            option: candidates[optionIndex],
            color: colors[optionIndex]
        };
    };
    
    return {
        rotate,
        getOptionByStep,
        getColorByStep
    };
}

function getRandomColor() {
  const letters = '0123456789ABCDEF'.split('');
  let color = '';
  for (let i = 0; i < 6; i++) {
    color += letters[Math.floor(Math.random() * 16)];
  }
  return `#${color}`;
}

function get_color(length:number) {
    const colors = [];
    for (let i = 0; i < length; i++) {
        colors.push(getRandomColor());
    }
    return colors;
}

function get_background(background: string): string {
    const bg = parseInt(background.substr(1), 16);
    const r = (bg >> 16) & 0xff;
    const g = (bg >> 8) & 0xff;
    const b = bg & 0xff;
    const luma = 0.2126 * r + 0.7152 * g + 0.0722 * b;
    return luma > 130 ? '#000' : '#fff';
}