/**
 * Copyright 2024 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import { StreamingLog } from "../multimodal-live-types";

interface StoreLoggerState {
  maxLogs: number;
  logs: StreamingLog[];
  log: (streamingLog: StreamingLog) => void;
  clearLogs: () => void;
  setMaxLogs: (n: number) => void;
}

class LoggerStore implements StoreLoggerState {
  maxLogs = 500;
  logs: StreamingLog[] = [];

  log({ date, type, message }: StreamingLog) {
    const prevLog = this.logs.at(-1);
    if (prevLog && prevLog.type === type && prevLog.message === message) {
      this.logs = [
        ...this.logs.slice(0, -1),
        {
          date,
          type,
          message,
          count: prevLog.count ? prevLog.count + 1 : 1,
        } as StreamingLog,
      ];
    } else {
      this.logs = [
        ...this.logs.slice(-(this.maxLogs - 1)),
        {
          date,
          type,
          message,
        } as StreamingLog,
      ];
    }
  }

  clearLogs() {
    console.log("clear log");
    this.logs = [];
  }

  setMaxLogs(n: number) {
    this.maxLogs = n;
  }
}

export const loggerStore = new LoggerStore();
