/**
 * Copyright 2024 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import { audioContext } from "./utils";
import AudioRecordingWorklet from "./worklets/audio-processing";
import VolMeterWorket from "./worklets/vol-meter";
import { createWorketFromSrc } from "./audioworklet-registry";
import EventEmitter from "eventemitter3";

// Node.js 환경에서 필요한 모듈을 가져옵니다.
import { Readable } from "stream";
import { AudioContext as NodeAudioContext } from "node-web-audio-api";
const mic = require("mic");

function arrayBufferToBase64(buffer: ArrayBuffer) {
  var binary = "";
  var bytes = new Uint8Array(buffer);
  var len = bytes.byteLength;
  for (var i = 0; i < len; i++) {
    binary += String.fromCharCode(bytes[i]);
  }
  return btoa(binary);
}

export class AudioRecorder extends EventEmitter {
  stream: MediaStream | undefined;
  audioContext: AudioContext | undefined;
  source: MediaStreamAudioSourceNode | undefined;
  recording: boolean = false;
  recordingWorklet: AudioWorkletNode | undefined;
  vuWorklet: AudioWorkletNode | undefined;
  micInstance: any;
  micInputStream: Readable | undefined;

  private starting: Promise<void> | null = null;

  constructor(public sampleRate = 16000) {
    super();
  }

  async start() {
    if (typeof window === "undefined") {
      // Node.js 환경
      this.starting = new Promise(async (resolve, reject) => {
        try {
          this.audioContext = new NodeAudioContext({ sampleRate: this.sampleRate });
          const workletName = "audio-recorder-worklet";
          const src = createWorketFromSrc(workletName, AudioRecordingWorklet);

          await this.audioContext.audioWorklet.addModule(src);
          this.recordingWorklet = new AudioWorkletNode(this.audioContext, workletName);

          this.recordingWorklet.port.onmessage = async (ev: MessageEvent) => {
            const arrayBuffer = ev.data.data.int16arrayBuffer;
            if (arrayBuffer) {
              const arrayBufferString = arrayBufferToBase64(arrayBuffer);
              this.emit("data", arrayBufferString);
            }
          };

          // Node.js 환경에서는 mic 패키지를 사용하여 실제 마이크 입력을 처리합니다.
          this.micInstance = mic({
            rate: this.sampleRate.toString(),
            channels: "1",
            debug: false,
            exitOnSilence: 6,
          });

          this.micInputStream = this.micInstance.getAudioStream();
          this.micInputStream?.on("data", (data: Buffer) => {
            const float32Array = new Float32Array(data.length / 2);
            const dataView = new DataView(data.buffer);

            for (let i = 0; i < data.length / 2; i++) {
              const int16 = dataView.getInt16(i * 2, true);
              float32Array[i] = int16 / 32768;
            }

            this.recordingWorklet?.port.postMessage({
              event: "chunk",
              data: {
                int16arrayBuffer: float32Array.buffer,
              },
            });
          });

          this.micInstance.start();

          // vu meter worklet
          const vuWorkletName = "vu-meter";
          await this.audioContext.audioWorklet.addModule(createWorketFromSrc(vuWorkletName, VolMeterWorket));
          this.vuWorklet = new AudioWorkletNode(this.audioContext, vuWorkletName);
          this.vuWorklet.port.onmessage = (ev: MessageEvent) => {
            this.emit("volume", ev.data.volume);
          };

          this.recording = true;
          resolve();
          this.starting = null;
        } catch (error) {
          reject(error);
        }
      });
    } else {
      // 브라우저 환경
      this.starting = new Promise(async (resolve, reject) => {
        try {
          this.stream = await navigator.mediaDevices.getUserMedia({ audio: true });
          this.audioContext = await audioContext({ sampleRate: this.sampleRate });
          this.source = this.audioContext.createMediaStreamSource(this.stream);

          const workletName = "audio-recorder-worklet";
          const src = createWorketFromSrc(workletName, AudioRecordingWorklet);

          await this.audioContext.audioWorklet.addModule(src);
          this.recordingWorklet = new AudioWorkletNode(this.audioContext, workletName);

          this.recordingWorklet.port.onmessage = async (ev: MessageEvent) => {
            const arrayBuffer = ev.data.data.int16arrayBuffer;
            if (arrayBuffer) {
              const arrayBufferString = arrayBufferToBase64(arrayBuffer);
              this.emit("data", arrayBufferString);
            }
          };
          this.source.connect(this.recordingWorklet);

          // vu meter worklet
          const vuWorkletName = "vu-meter";
          await this.audioContext.audioWorklet.addModule(createWorketFromSrc(vuWorkletName, VolMeterWorket));
          this.vuWorklet = new AudioWorkletNode(this.audioContext, vuWorkletName);
          this.vuWorklet.port.onmessage = (ev: MessageEvent) => {
            this.emit("volume", ev.data.volume);
          };

          this.source.connect(this.vuWorklet);
          this.recording = true;
          resolve();
          this.starting = null;
        } catch (error) {
          reject(error);
        }
      });
    }
  }

  stop() {
    const handleStop = () => {
      this.source?.disconnect();
      if (this.stream) {
        this.stream.getTracks().forEach((track) => track.stop());
      }
      if (this.micInstance) {
        this.micInstance.stop();
      }
      this.stream = undefined;
      this.recordingWorklet = undefined;
      this.vuWorklet = undefined;
    };
    if (this.starting) {
      this.starting.then(handleStop);
      return;
    }
    handleStop();
  }
}
